#pragma once
/******************************************************
MOUSEBGI.H
Routines to control the mouse, adapted to WinBGI

Developed by:

	JOSE LUIS DE LA CRUZ LAZARO
	contact@theworldofchaos.com

	YACSHA - Software & Desing, since 1999, Lima-Peru
	>> The World of Chaos - EL MUNDO DEL CAOS - Unlimited Programming
	www.theworldofchaos.com

LICENSE
	This file is released under the terms of the BSD 3-Clause License.
	Read LICENSE.txt file for more information.

HISTORY...

 >> Version 3 - 12-IV-2024
	- Update credits
	- Translate variables name, credits and comments
	  from spanish to english

 >> Version 2 - 24-III-2024
	- Update graphics/MouseBGI - Porting to VC++ 2017
	  using winbgi
	- The entire library is migrated to the MouseBGI class,
	  using the mouse functionalities of WinBGI
	- All code from version 1.x is discarded, maintaining the
	  declarations of the main functions (mouseevent and
	  Detect_click_or_key), for better backward compatibility
	  with programs that used previous versions of this library;
	  and new code is developed for the implementations,
	  from scratch, that is compatible with the
	  WinBGI mouse functions

 >> Version 1.1a - 06-VIII-2000
	- Almost a year has passed since Silvia gave me the
	  original source of the present. And well, that
	  deserves further improvements...
	- The standard cursor is now supported in
	  320x200 256 colors graphic mode, and also the standard
	  cursor in text mode.
	- Added the original structures to load standard cursors
	  (arrowcursor, handcursor and cross)
	- The mousesetup function is modified to select the graphic
	  mode and the type of standard cursor to use
	  (by default it is the arrowcursor)

 >> Version 1.1 - 28-VI-2000
	- The pointer is deleted after drawing it temporarily to
	  load it into memory (see get_cursor function)

 >> Version 1.0b - 28-XII-1999
	- A 5 millisecond pause is added before calling the function
	  "update_pointer", this is done to move the mouse pointer
	  smoothly across the screen, before this the pointer moved
	  with some pauses. All this takes place in the implementation
	  of the "mousestatus" function.

 >> Version 1.0a - 16-XI-1999
	- Detailed comments on the operation of all functions are added

 >> Version 1.0 - 11-IX-1999
	- Modification of the original routines corresponding to the
	  mousec.h header that works only with the graphic mode in
	  16 colors
	- It works perfectly in the 256-color graphic mode using the
	  bgi svga256.bgi driver, this is done through the control
	  functions for the bgi created for this purpose.

 >> Version 0 - 11-VIII-1999
	- Original sources extracted from MOUSEC.H
	- thanks to Silvia Garcia - August 16, 1999

*******************************************************/

#include "graphics.h"
#include <vector>

class MouseBGI {

public:
	enum class EVENT {
		////////////////////////////////////////////////// //////////////////////
		//CONTROL CONSTANTS
		// mouse event codes
		IDLE, //no events
		LBUTTON_DOWN, //the left mouse button was pressed
		RBUTTON_DOWN //the right mouse button was pressed
	};

	// Structure that stores the events generated by the mouse and keyboard
	struct Event
	{
		int x; //x position of the mouse pointer
		int y; //position and of the mouse pointer
		MouseBGI::EVENT event; //Current mouse event (see definition of Event codes below)
		char key; //if a key is pressed from the keyboard, its ASCII code is stored here
	};


public:
	MouseBGI();
	~MouseBGI();

	// Mouse handler for WinBGI mouse controller
	static void OnLButtonDown(int x, int y);
	static void OnRButtonDown(int x, int y);
	static void OnMButtonDown(int x, int y);
	static void OnLButtonUp(int x, int y);
	static void OnRButtonUp(int x, int y);
	static void OnMButtonUp(int x, int y);
	static void OnMouseMove(int x, int y);

	static void RegisterMouseEvents();

	// obtiene el ultimo evento del raton
	static EVENT mouseevent(int&, int&);

	static Event Detect_click_or_key();	

private:
	// Variables for monitoring mouse clicks
	static bool bLBUTTONPRESSED;
	static bool bRBUTTONPRESSED;
	static bool bMOUSEMOVE;

};

MouseBGI::MouseBGI() {
}

MouseBGI::~MouseBGI() {
}

bool MouseBGI::bLBUTTONPRESSED = false;
bool MouseBGI::bRBUTTONPRESSED = false;
bool MouseBGI::bMOUSEMOVE = false;

///////////////////////////////////////////////////////////////////
// STANDARD CONTROL FUNCTIONS

void MouseBGI::OnLButtonDown(int x, int y) {
	bLBUTTONPRESSED = true;
}

void MouseBGI::OnRButtonDown(int x, int y) {
	bRBUTTONPRESSED = true;
}

void MouseBGI::OnMButtonDown(int x, int y) {
}

void MouseBGI::OnLButtonUp(int x, int y) {
	bLBUTTONPRESSED = false;
}

void MouseBGI::OnRButtonUp(int x, int y) {
	bRBUTTONPRESSED = false;
}

void MouseBGI::OnMButtonUp(int x, int y) {
}

void MouseBGI::OnMouseMove(int x, int y) {

}

void MouseBGI::RegisterMouseEvents() {
	registermousehandler(WM_LBUTTONDOWN, MouseBGI::OnLButtonDown);
	registermousehandler(WM_RBUTTONDOWN, MouseBGI::OnRButtonDown);
	registermousehandler(WM_MBUTTONDOWN, MouseBGI::OnMButtonDown);
	registermousehandler(WM_LBUTTONUP, MouseBGI::OnLButtonUp);
	registermousehandler(WM_RBUTTONUP, MouseBGI::OnRButtonUp);
	registermousehandler(WM_MBUTTONUP, MouseBGI::OnMButtonUp);
	registermousehandler(WM_MOUSEMOVE, MouseBGI::OnMouseMove);
}

////////////////////////////////////////////////////////////////////////////
// get the last mouse event
MouseBGI::EVENT MouseBGI::mouseevent(int& Mx, int& My) {
	EVENT E;

	if (bLBUTTONPRESSED) {
		E = EVENT::LBUTTON_DOWN;
		//cleardevice();
		//outtext("L");			
	}
	else if (bRBUTTONPRESSED) {
		E = EVENT::RBUTTON_DOWN;
		//cleardevice();
		//outtext("R");

	}
	else
		E = EVENT::IDLE;

	Mx = mousex();
	My = mousey();
	
	return E;
}

/////////////////////////////////////
// IMPLEMENTATION OF EVENTS
MouseBGI::Event MouseBGI::Detect_click_or_key()
{
	Event p;

	p.event = EVENT::IDLE;
	p.key = -1;

	p.event = mouseevent(p.x, p.y);

	if (kbhit())
	{
		p.key = getch();
	}

	return p;
}


